/*
 * Decompiled with CFR 0.152.
 */
package com.zaxxer.hikari.pool;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.pool.HikariPool;
import com.zaxxer.hikari.pool.PoolBagEntry;
import com.zaxxer.hikari.util.DefaultThreadFactory;
import com.zaxxer.hikari.util.DriverDataSource;
import com.zaxxer.hikari.util.PropertyElf;
import com.zaxxer.hikari.util.UtilityElf;
import java.lang.management.ManagementFactory;
import java.lang.reflect.Field;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PoolElf {
    private static final Logger LOGGER = LoggerFactory.getLogger(PoolElf.class);
    private static final String[] RESET_STATES = new String[]{"readOnly", "autoCommit", "isolation", "catalog", "netTimeout"};
    private static final int UNINITIALIZED = -1;
    private static final int TRUE = 1;
    private static final int FALSE = 0;
    private int networkTimeout;
    private int transactionIsolation;
    private long validationTimeout;
    private int isNetworkTimeoutSupported;
    private int isQueryTimeoutSupported;
    private Executor netTimeoutExecutor;
    private final HikariConfig config;
    private final String poolName;
    private final String catalog;
    private final Boolean isReadOnly;
    private final boolean isAutoCommit;
    private final boolean isUseJdbc4Validation;
    private final boolean isIsolateInternalQueries;
    private volatile boolean isValidChecked;
    private volatile boolean isValidSupported;

    public PoolElf(HikariConfig configuration) {
        this.config = configuration;
        this.networkTimeout = -1;
        this.catalog = this.config.getCatalog();
        this.isReadOnly = this.config.isReadOnly();
        this.isAutoCommit = this.config.isAutoCommit();
        this.validationTimeout = this.config.getValidationTimeout();
        this.transactionIsolation = PoolElf.getTransactionIsolation(this.config.getTransactionIsolation());
        this.isValidSupported = true;
        this.isQueryTimeoutSupported = -1;
        this.isNetworkTimeoutSupported = -1;
        this.isUseJdbc4Validation = this.config.getConnectionTestQuery() == null;
        this.isIsolateInternalQueries = this.config.isIsolateInternalQueries();
        this.poolName = this.config.getPoolName();
    }

    public void quietlyCloseConnection(Connection connection, String closureReason) {
        try {
            if (connection == null || connection.isClosed()) {
                return;
            }
            LOGGER.debug("{} - Closing connection {}: {}", new Object[]{this.poolName, connection, closureReason});
            try {
                this.setNetworkTimeout(connection, TimeUnit.SECONDS.toMillis(15L));
            }
            finally {
                connection.close();
            }
        }
        catch (Throwable e) {
            LOGGER.debug("{} - Closing connection {} failed", new Object[]{this.poolName, connection, e});
        }
    }

    public static int getTransactionIsolation(String transactionIsolationName) {
        if (transactionIsolationName != null) {
            try {
                Field field = Connection.class.getField(transactionIsolationName);
                return field.getInt(null);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid transaction isolation value: " + transactionIsolationName);
            }
        }
        return -1;
    }

    DataSource initializeDataSource() {
        String jdbcUrl = this.config.getJdbcUrl();
        String username = this.config.getUsername();
        String password = this.config.getPassword();
        String dsClassName = this.config.getDataSourceClassName();
        String driverClassName = this.config.getDriverClassName();
        Properties dataSourceProperties = this.config.getDataSourceProperties();
        DataSource dataSource = this.config.getDataSource();
        if (dsClassName != null && dataSource == null) {
            dataSource = UtilityElf.createInstance(dsClassName, DataSource.class, new Object[0]);
            PropertyElf.setTargetFromProperties(dataSource, dataSourceProperties);
        } else if (jdbcUrl != null && dataSource == null) {
            dataSource = new DriverDataSource(jdbcUrl, driverClassName, dataSourceProperties, username, password);
        }
        if (dataSource != null) {
            this.setLoginTimeout(dataSource, this.config.getConnectionTimeout());
            this.createNetworkTimeoutExecutor(dataSource, dsClassName, jdbcUrl);
        }
        return dataSource;
    }

    void setupConnection(Connection connection, long connectionTimeout) throws SQLException {
        if (this.isUseJdbc4Validation && !this.isJdbc4ValidationSupported(connection)) {
            throw new SQLException("Connection.isValid() method is not supported, connection test query must be configured");
        }
        this.networkTimeout = this.getAndSetNetworkTimeout(connection, connectionTimeout);
        this.transactionIsolation = this.transactionIsolation < 0 ? connection.getTransactionIsolation() : this.transactionIsolation;
        connection.setAutoCommit(this.isAutoCommit);
        if (this.isReadOnly != null) {
            connection.setReadOnly(this.isReadOnly);
        }
        if (this.transactionIsolation != connection.getTransactionIsolation()) {
            connection.setTransactionIsolation(this.transactionIsolation);
        }
        if (this.catalog != null) {
            connection.setCatalog(this.catalog);
        }
        this.executeSql(connection, this.config.getConnectionInitSql(), this.isAutoCommit);
        this.setNetworkTimeout(connection, this.networkTimeout);
    }

    boolean isConnectionAlive(Connection connection, AtomicReference<Throwable> lastConnectionFailure) {
        try {
            int timeoutSec = (int)TimeUnit.MILLISECONDS.toSeconds(this.validationTimeout);
            if (this.isUseJdbc4Validation) {
                return connection.isValid(timeoutSec);
            }
            int originalTimeout = this.getAndSetNetworkTimeout(connection, this.validationTimeout);
            try (Statement statement = connection.createStatement();){
                this.setQueryTimeout(statement, timeoutSec);
                if (statement.execute(this.config.getConnectionTestQuery())) {
                    statement.getResultSet().close();
                }
            }
            if (this.isIsolateInternalQueries && !this.isAutoCommit) {
                connection.rollback();
            }
            this.setNetworkTimeout(connection, originalTimeout);
            return true;
        }
        catch (SQLException e) {
            lastConnectionFailure.set(e);
            LOGGER.warn("{} - Connection {} failed alive test with exception {}", new Object[]{this.poolName, connection, e.getMessage()});
            return false;
        }
    }

    void resetConnectionState(PoolBagEntry poolEntry) throws SQLException {
        String currentCatalog;
        int resetBits = 0;
        if (this.isReadOnly != null && poolEntry.isReadOnly != this.isReadOnly) {
            poolEntry.connection.setReadOnly(this.isReadOnly);
            resetBits |= 1;
        }
        if (poolEntry.isAutoCommit != this.isAutoCommit) {
            poolEntry.connection.setAutoCommit(this.isAutoCommit);
            resetBits |= 2;
        }
        if (poolEntry.transactionIsolation != this.transactionIsolation) {
            poolEntry.connection.setTransactionIsolation(this.transactionIsolation);
            resetBits |= 4;
        }
        if ((currentCatalog = poolEntry.catalog) != null && !currentCatalog.equals(this.catalog) || currentCatalog == null && this.catalog != null) {
            poolEntry.connection.setCatalog(this.catalog);
            resetBits |= 8;
        }
        if (poolEntry.networkTimeout != this.networkTimeout) {
            this.setNetworkTimeout(poolEntry.connection, this.networkTimeout);
            resetBits |= 0x10;
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("{} - Reset ({}) on connection {}", (Object)(resetBits != 0 ? this.stringFromResetBits(resetBits) : "nothing"), (Object)poolEntry.connection);
        }
    }

    void resetPoolEntry(PoolBagEntry poolEntry) {
        if (this.isReadOnly != null) {
            poolEntry.setReadOnly(this.isReadOnly);
        }
        poolEntry.setCatalog(this.catalog);
        poolEntry.setAutoCommit(this.isAutoCommit);
        poolEntry.setNetworkTimeout(this.networkTimeout);
        poolEntry.setTransactionIsolation(this.transactionIsolation);
    }

    void setValidationTimeout(long validationTimeout) {
        this.validationTimeout = validationTimeout;
    }

    void registerMBeans(HikariPool pool) {
        if (!this.config.isRegisterMbeans()) {
            return;
        }
        try {
            MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
            ObjectName beanConfigName = new ObjectName("com.zaxxer.hikari:type=PoolConfig (" + this.poolName + ")");
            ObjectName beanPoolName = new ObjectName("com.zaxxer.hikari:type=Pool (" + this.poolName + ")");
            if (!mBeanServer.isRegistered(beanConfigName)) {
                mBeanServer.registerMBean(this.config, beanConfigName);
                mBeanServer.registerMBean(pool, beanPoolName);
            } else {
                LOGGER.error("{} - You cannot use the same pool name for separate pool instances.", (Object)this.poolName);
            }
        }
        catch (Exception e) {
            LOGGER.warn("{} - Unable to register management beans.", (Object)this.poolName, (Object)e);
        }
    }

    void unregisterMBeans() {
        if (!this.config.isRegisterMbeans()) {
            return;
        }
        try {
            MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
            ObjectName beanConfigName = new ObjectName("com.zaxxer.hikari:type=PoolConfig (" + this.poolName + ")");
            ObjectName beanPoolName = new ObjectName("com.zaxxer.hikari:type=Pool (" + this.poolName + ")");
            if (mBeanServer.isRegistered(beanConfigName)) {
                mBeanServer.unregisterMBean(beanConfigName);
                mBeanServer.unregisterMBean(beanPoolName);
            }
        }
        catch (Exception e) {
            LOGGER.warn("{} - Unable to unregister management beans.", (Object)this.poolName, (Object)e);
        }
    }

    void shutdownTimeoutExecutor() {
        if (this.netTimeoutExecutor != null && this.netTimeoutExecutor instanceof ThreadPoolExecutor) {
            ((ThreadPoolExecutor)this.netTimeoutExecutor).shutdownNow();
        }
    }

    private boolean isJdbc4ValidationSupported(Connection connection) {
        if (!this.isValidChecked) {
            try {
                connection.isValid(1);
            }
            catch (Throwable e) {
                this.isValidSupported = false;
                LOGGER.debug("{} - Connection.isValid() is not supported ({})", (Object)this.poolName, (Object)e.getMessage());
            }
            this.isValidChecked = true;
        }
        return this.isValidSupported;
    }

    private void setQueryTimeout(Statement statement, int timeoutSec) {
        block3: {
            if (this.isQueryTimeoutSupported != 0) {
                try {
                    statement.setQueryTimeout(timeoutSec);
                    this.isQueryTimeoutSupported = 1;
                }
                catch (Throwable e) {
                    if (this.isQueryTimeoutSupported != -1) break block3;
                    this.isQueryTimeoutSupported = 0;
                    LOGGER.debug("{} - Statement.setQueryTimeout() is not supported ({})", (Object)this.poolName, (Object)e.getMessage());
                }
            }
        }
    }

    private int getAndSetNetworkTimeout(Connection connection, long timeoutMs) {
        block3: {
            if (this.isNetworkTimeoutSupported != 0) {
                try {
                    int originalTimeout = connection.getNetworkTimeout();
                    connection.setNetworkTimeout(this.netTimeoutExecutor, (int)timeoutMs);
                    this.isNetworkTimeoutSupported = 1;
                    return originalTimeout;
                }
                catch (Throwable e) {
                    if (this.isNetworkTimeoutSupported != -1) break block3;
                    this.isNetworkTimeoutSupported = 0;
                    LOGGER.debug("{} - Connection.setNetworkTimeout() is not supported ({})", (Object)this.poolName, (Object)e.getMessage());
                }
            }
        }
        return 0;
    }

    private void setNetworkTimeout(Connection connection, long timeoutMs) throws SQLException {
        if (this.isNetworkTimeoutSupported == 1) {
            connection.setNetworkTimeout(this.netTimeoutExecutor, (int)timeoutMs);
        }
    }

    private void executeSql(Connection connection, String sql, boolean isAutoCommit) throws SQLException {
        if (sql != null) {
            try (Statement statement = connection.createStatement();){
                if (statement.execute(sql)) {
                    statement.getResultSet().close();
                }
                if (!isAutoCommit) {
                    connection.commit();
                }
            }
        }
    }

    private void createNetworkTimeoutExecutor(DataSource dataSource, String dsClassName, String jdbcUrl) {
        if (dsClassName != null && dsClassName.contains("Mysql") || jdbcUrl != null && jdbcUrl.contains("mysql") || dataSource != null && dataSource.getClass().getName().contains("Mysql")) {
            this.netTimeoutExecutor = new SynchronousExecutor();
        } else {
            ThreadFactory threadFactory = this.config.getThreadFactory() != null ? this.config.getThreadFactory() : new DefaultThreadFactory("Hikari JDBC-timeout executor", true);
            ThreadPoolExecutor executor = (ThreadPoolExecutor)Executors.newCachedThreadPool(threadFactory);
            executor.allowCoreThreadTimeOut(true);
            executor.setKeepAliveTime(15L, TimeUnit.SECONDS);
            this.netTimeoutExecutor = executor;
        }
    }

    private void setLoginTimeout(DataSource dataSource, long connectionTimeout) {
        if (connectionTimeout != Integer.MAX_VALUE) {
            try {
                dataSource.setLoginTimeout((int)TimeUnit.MILLISECONDS.toSeconds(Math.max(1000L, connectionTimeout)));
            }
            catch (SQLException e) {
                LOGGER.warn("{} - Unable to set DataSource login timeout", (Object)this.poolName, (Object)e);
            }
        }
    }

    private String stringFromResetBits(int bits) {
        StringBuilder sb = new StringBuilder();
        for (int ndx = 0; ndx < RESET_STATES.length; ++ndx) {
            if ((bits & 1 << ndx) == 0) continue;
            sb.append(RESET_STATES[ndx]).append(", ");
        }
        sb.setLength(sb.length() - 2);
        return sb.toString();
    }

    private static class SynchronousExecutor
    implements Executor {
        private SynchronousExecutor() {
        }

        @Override
        public void execute(Runnable command) {
            try {
                command.run();
            }
            catch (Throwable t) {
                LOGGER.debug("Exception executing {}", (Object)command, (Object)t);
            }
        }
    }
}

